/******************************************************************************
 [Module Name]: Ir.h
 [Date]:   02/May/2008
 [Comment]:
   Remote control header file.
 [Reversion History]:
*******************************************************************************/
#ifndef _IR_H
#define _IR_H

#include "ir_define.h"

#ifdef _IR_C_
#define _IRDEC_
#else
#define _IRDEC_ extern
#endif

//////////////////////////////////////
// Definition
//////////////////////////////////////

#define ISR_TIMER0_INTERVAL     0.550       // unit: ms

#define MCU_XTAL_CLK_HZ     22118400ul
#define MCU_XTAL_CLK_KHZ    (MCU_XTAL_CLK_HZ / 1000)
#define MCU_XTAL_CLK_MHZ    (MCU_XTAL_CLK_KHZ / 1000)


//////////////////////////////////////

#define pinIr    P4 ^ 2

//////////////////////////////////////
#define IR_HEADER_CODE0  0x08
#define IR_HEADER_CODE1  0xF6
//////////////////////////////////////
//    IR command delay time
//////////////////////////////////////
#define IR_CYCLE_TIME           (108+8)/ISR_TIMER0_INTERVAL               // ms(total ir cycle time)

#define IR_1ST_LEAD_TIME        13.5   // us(9+4.5 ms)
#define IR_REPAEAT_LEAD_TIME    11.5   // us(9+2.5 ms)

#define IR_LOGIC0_TIME          1.12   // us(0.56+0.56 ms)
#define IR_LOGIC1_TIME          2.24   // us(0.56+1.68 ms)

#define IR_TOLERANCE_TIME       ((IR_LOGIC1_TIME-IR_LOGIC0_TIME)/2)

#define IR_BYTES_NUM            4

// MCU timer for IR using
#define IR_TIMER_INTERVAL           20 // unit: 1ms

#define IrCalculateCounter(time)        (((MCU_XTAL_CLK_KHZ*(time))+6)/12)  // +6 for |ˤJ
#define IR_TIMER_COUNTER                (65536-IrCalculateCounter(IR_TIMER_INTERVAL))
#define irGetMinCnt(time)               IrCalculateCounter((time-IR_TOLERANCE_TIME))
#define irGetMaxCnt(time)               IrCalculateCounter((time+IR_TOLERANCE_TIME))

typedef enum _IrCommandType
{
    IRKEY_AD           = 0x07,
    IRKEY_PA           = 0x13,
    IRKEY_RS           = 0x02,
    IRKEY_IO           = 0x16,
    IRKEY_0            = 0x01,
    IRKEY_1            = 0x12,
    IRKEY_2            = 0x0B,
    IRKEY_3            = 0x05,
    IRKEY_4            = 0x0D,
    IRKEY_5            = 0x0F,
    IRKEY_6            = 0x0E,
    IRKEY_7            = 0x08,
    IRKEY_8            = 0x0C,
    IRKEY_9            = 0x15,
    IRKEY_ESC          = 0x04,
    IRKEY_MD           = 0x03,

    IRKEY_100          = 0x55,

    IRKEY_POWER        = 0x5C,
    IRKEY_INPUT        = 0x42,

    IRKEY_OSD_MENU     = 0x57,
    IRKEY_OSD_LEFT     = 0x52,
    IRKEY_OSD_RIGHT    = 0x46,
    IRKEY_OSD_UP       = 0x3E,
    IRKEY_OSD_DOWM     = 0x50,

    IRKEY_VOL_UP       = 0x43,
    IRKEY_VOL_DOWM     = 0x4E,
    IRKEY_CH_UP        = 0x41,
    IRKEY_CH_DOWM      = 0x48,

    IRKEY_FUNCTION     = 0x5A,

    IRKEY_MUTE         = 0x49,
    IRKEY_MTS          = 0xFB,
    IRKEY_DISPLAY      = 0x60,
    IRKEY_SLEEP        = 0x59,
    IRKEY_ZOOM         = 0xFE,
    IRKEY_PICMODE      = 0x4D,
    IRKEY_SOUNDMODE    = 0x4B,
    IRKEY_SCANMODE     = 0xFD,

    IRKEY_LAST_CH      = 0x53,
    IRKEY_DUMY         = 0xFF
}IrCommandType;

//////////////////////////////////////////////////////
#define IR_DELAY_TIME  (3000 / ISR_TIMER0_INTERVAL) // IR command start in INT1 to IR command end time
#define IR_DELAY_TIME0 (500 / ISR_TIMER0_INTERVAL)  // IR 1st command to repeat command delay time
#define IR_DELAY_TIME1 (100 / ISR_TIMER0_INTERVAL)  // IR command hold time after last repeat command
//////////////////////////////////////////////////////
#define IR_REPEAT_START_TIME    (500 / ISR_TIMER0_INTERVAL)
#define IR_REPEAT_END_TIME      (550 / ISR_TIMER0_INTERVAL)
//////////////////////////////////////////////////////
// tuner change channel delay time
#define IR_NUMKEY_DELAY_TIME1 (3000 / ISR_TIMER0_INTERVAL) // IR 0-9 key press ending delay time (1st key)
#define IR_NUMKEY_DELAY_TIME2 (1000 / ISR_TIMER0_INTERVAL) // IR 0-9 key press ending delay time (2,3 key)
//////////////////////////////////////////////////////////////////
// Variables
//////////////////////////////////////////////////////////////////
// IR recieve action
_IRDEC_ bit g_bIrError;   // IR command receive error flag
_IRDEC_ bit g_bIrDetect;  // IR command detect flag
_IRDEC_ bit g_bIrCommand; // IR command input hold flag
_IRDEC_ bit g_bIrExecute; // IR command 1st execute busy flag
_IRDEC_ bit g_bIrTime;    // IR command end & start delay timer

_IRDEC_ bit g_bIrRepeat;  // IR command repeat flag
_IRDEC_ bit g_bIrCheckRepeat; // check repeat in MCU Timer0(delay to repeat)
_IRDEC_ bit g_bIrRepeatStatus;  // IR command repeat status

_IRDEC_ BYTE g_ucIrBitsCnt; // IR bits counter
_IRDEC_ WORD g_wIrPeriodCnt; // IR single period counter
_IRDEC_ BYTE g_ucIrDataFifo; // IR data FIFO

_IRDEC_ XDATA WORD g_wIrCycleTimer; // IR total cycle timer
_IRDEC_ XDATA BYTE g_ucAddresCode0; // IR address code 0
_IRDEC_ XDATA BYTE g_ucAddresCode1; // IR address code 1
_IRDEC_ XDATA BYTE g_ucDataCode; // IR data code

_IRDEC_ XDATA BYTE g_ucIrCode; // IR code
_IRDEC_ XDATA WORD g_wIrTimer; // IR timer

// IR key
_IRDEC_ XDATA BOOL g_bIrNumKeyStart; // IR 0-9 key pressed start flag
_IRDEC_ bit g_bIrKeyNumEnd;   // IR 0-9 key pressed end flag

_IRDEC_ bit g_bIr100Key; // pressed +100 key

_IRDEC_ XDATA WORD g_ucIrNumKey;      // IR number key
_IRDEC_ XDATA WORD g_wIrNumKeyTimer; // IR number key timer

//////////////////////////////////////
// Subroutines
//////////////////////////////////////
_IRDEC_ void irInterruptProcess (void);
_IRDEC_ void irDetectTimer0 (void);
_IRDEC_ void irDetectTimer1 (void);

_IRDEC_ void irDecodeRepaeat (WORD wRepeatTime);
_IRDEC_ void irDecodeEnd (void);

_IRDEC_ unsigned char irDecodeCommand (void);
_IRDEC_ unsigned char IRKeyToNumber (BYTE ucCode);
#endif
